<?php
/**
 * Plugin Name: Activation Code Plugin
 * Description: Custom post type for activation codes.
 * Version: 1.0
 * Author: Your Name
 */

// Register Activation Code Custom Post Type
function register_activation_code_post_type() {
    $labels = array(
        'name'               => 'Activation Codes',
        'singular_name'      => 'Activation Code',
        'menu_name'          => 'Activation Codes',
        'add_new'            => 'Add New',
        'add_new_item'       => 'Add New Activation Code',
        'edit_item'          => 'Edit Activation Code',
        'new_item'           => 'New Activation Code',
        'view_item'          => 'View Activation Code',
        'search_items'       => 'Search Activation Codes',
        'not_found'          => 'No activation codes found',
        'not_found_in_trash' => 'No activation codes found in trash',
    );

    $args = array(
        'labels'              => $labels,
        'public'              => false,
        'exclude_from_search' => true,
        'publicly_queryable'  => false,
        'show_ui'             => true,
        'show_in_menu'        => true,
        'menu_icon'           => 'dashicons-key',
        'capability_type'     => 'post',
        'hierarchical'        => false,
        'supports'            => array( 'title' ),
    );

    register_post_type( 'activation_code', $args );
}

add_action( 'init', 'register_activation_code_post_type' );

// Add Expiry Date and Limit Fields to Activation Code Post Type
function add_custom_fields() {
    add_meta_box( 'activation_code_meta_box', 'Activation Code Details', 'display_activation_code_meta_box', 'activation_code', 'normal', 'high' );
    add_meta_box( 'urls_meta_box', 'Active Website', 'display_urls_meta_box', 'activation_code', 'normal', 'high' );
}

add_action( 'add_meta_boxes', 'add_custom_fields' );

function display_activation_code_meta_box( $post ) {
    // Retrieve existing values for fields
    $expiry_date = get_post_meta( $post->ID, '_expiry_date', true );
    $limit = get_post_meta( $post->ID, '_limit', true );
    $activation_code = get_post_meta( $post->ID, '_activation_code', true );

    // Output fields
    ?>
    <label for="expiry_date" style="padding: 10px;width: 30%;display: inline-block;">Expiry Date:</label>
    <input type="date" style="min-width: 220px;" name="expiry_date" id="expiry_date" value="<?php echo esc_attr( $expiry_date ); ?>" required>
    <br>

    <label for="limit" style="padding: 10px;width: 30%;display: inline-block;">Limit:</label>
    <input type="number" style="min-width: 220px;" name="limit" id="limit" value="<?php echo esc_attr( $limit ); ?>" required>
    <br>

    <label for="activation_code" style="padding: 10px;width: 30%;display: inline-block;">Activation Code:</label>
    <input type="text" style="min-width: 220px;" name="activation_code" id="activation_code"  value="<?php echo esc_attr( $activation_code ); ?>">
    
    <button type="button" id="generate_code">Generate Code</button>

    <script>
        document.getElementById('generate_code').addEventListener('click', function() {
            document.getElementById('activation_code').value = generateActivationCode();
        });

        function generateActivationCode() {
            const characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
            return Array.from({ length: 20 }, () => characters[Math.floor(Math.random() * characters.length)]).join('');
        }
    </script>
    <?php
}

function display_urls_meta_box( $post ) {
    // Output the URLs field as readonly
    $urlList = json_decode(get_post_meta( $post->ID, '_urls', true ));
    if($urlList != ""){        
    ?>
        <div><?php foreach($urlList as $url){ echo '<p>'.$url . '</p>'; }?></div>
    <?php }
}

// Save custom field values
function save_custom_fields( $post_id ) {
    if ( isset( $_POST['expiry_date'] ) ) {
        update_post_meta( $post_id, '_expiry_date', sanitize_text_field( $_POST['expiry_date'] ) );
    }

    if ( isset( $_POST['limit'] ) ) {
        update_post_meta( $post_id, '_limit', intval( $_POST['limit'] ) );
    }

    if ( isset( $_POST['activation_code'] ) ) {
        update_post_meta( $post_id, '_activation_code', $_POST['activation_code']);
    }
}

add_action( 'save_post', 'save_custom_fields' );

// Add custom API endpoint for activation key verification
function activation_key_verification_endpoint() {
    // Endpoint: websiteurl/accessKeyVerify/pluginActivationkey.php
    register_rest_route('activation-key-verify', '/pluginActivationkey/', array(
        'methods' => 'POST',
        'callback' => 'handle_activation_key_verification',
        'permission_callback' => '__return_true',
    ));
}

add_action('rest_api_init', 'activation_key_verification_endpoint');

// Callback function to handle activation key verification
function handle_activation_key_verification(WP_REST_Request $request) {
    $api_key = $request->get_param('api_key');
    $website_url = $request->get_param('website_url');
    $access_code = $request->get_param('access_code');

    if (empty($api_key) || empty($website_url)) {
        error_log('API key or website URL missing.');
        return new WP_REST_Response(array('status' => false, 'message' => 'API key and website URL are required.'));
    }

    $activation_code_query = new WP_Query(array(
        'post_type' => 'activation_code',
        'meta_query' => array(
            array(
                'key' => '_activation_code',
                'value' => $api_key,
            ),
        ),
    ));

    if ($activation_code_query->have_posts()) {
        while ($activation_code_query->have_posts()) {
            $activation_code_query->the_post();
            $post_id = get_the_ID();

            $urls = get_post_meta($post_id, '_urls', true);
            $limit = get_post_meta($post_id, '_limit', true);
            $expiry_date = get_post_meta($post_id, '_expiry_date', true);
            $urls_list = json_decode($urls);

            if (empty($urls) || strpos($urls, extractMainDomain($website_url)) !== false || count($urls_list) < $limit) {
                $current_date = current_time('mysql');
                if (strtotime($expiry_date) < strtotime($current_date)) {
                    error_log('Activation code expired.');
                    return new WP_REST_Response(array('status' => false, 'message' => 'Date Expired'));
                } else {
                    if (empty($urls)) {
                        $urls_data[] = $website_url;
                    } elseif (strpos($urls, extractMainDomain($website_url)) !== false) {
                        $urls_data = $urls_list;
                    } else {
                        $urls_data = $urls_list;
                        $urls_data[] = $website_url;
                    }
                    update_post_meta($post_id, '_urls', json_encode($urls_data));
                    error_log('Activation successful: ' . $api_key);
                    return new WP_REST_Response(array('status' => true, 'expiry_date' => $expiry_date));
                }
            } else {
                error_log('Activation code already used: ' . $api_key);
                return new WP_REST_Response(array('status' => false, 'message' => 'Activation code already used.'));
            }
        }
    }

    error_log('Invalid activation code: ' . $api_key);
    return new WP_REST_Response(array('status' => false, 'message' => 'Invalid activation code.'));
}

function extractMainDomain($url) {
    $host = parse_url($url, PHP_URL_HOST);
    $hostParts = explode('.', $host);
    $mainDomain = implode('.', array_slice($hostParts, -2)); // Extract the last two parts of the host
    return $mainDomain;
}

// WooCommerce Integration to generate Activation Codes per product
add_action('woocommerce_order_status_completed', 'generateActivationCodePerProduct');
add_action('woocommerce_order_status_processing', 'generateActivationCodePerProduct');

function generateActivationCodePerProduct($orderID){
    $order = wc_get_order($orderID);
    $customer_email = $order->get_billing_email();
    if (!$order) return;

    foreach ($order->get_items('line_item') as $item_id => $item) {
        $productData = json_decode($item);
        if (isset($productData) && !empty($productData)) {
            $name = $productData->name;
            $period = '';

            if ($name == 'Lifetime plan') {
                $period = 54787;
            } elseif ($name == 'Annual plan') {
                $period = 365;
            } elseif ($name == 'Monthly plan') {
                $period = 30;
            }

            if (!empty($period)) {
                $activationCode = wc_get_order_item_meta($item_id, '_activation_code');
                if (empty($activationCode)) {
                    $generatedCode = generateActivationCode($orderID, $period);
                    send_email($customer_email, sanitize_text_field($generatedCode), getExpireDate($period), $name);
                }
            }
        }
    }
}

function getExpireDate($expire) {
    $expire = date('Y-m-d', strtotime(date('Y-m-d').' + '.$expire.' days'));
    return $expire;
}

function generateActivationCode($order_id, $expire) {
    $expire = getExpireDate($expire);
    $post_id = wp_insert_post(array(
        'post_type' => 'activation_code',
        'post_title' => 'For Order #' . $order_id,
        'post_content' => '',
        'post_status' => 'publish',
    ));

    if ($post_id) {
        $generateCode = cdr__generateActivationCode();
        update_post_meta($post_id, '_expiry_date', $expire);
        update_post_meta($post_id, '_activation_code', $generateCode);
        update_post_meta($post_id, '_limit', 1);
    }
    return $generateCode;
}

function cdr__generateActivationCode() {
    $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $codeLength = 20;
    $activationCode = '';

    for ($i = 0; $i < $codeLength; $i++) {
        $randomIndex = rand(0, strlen($characters) - 1);
        $activationCode .= $characters[$randomIndex];
    }

    return $activationCode;
}

function send_email($customer_email, $activation_code, $expire_date, $name) {
    $year_expire = date('Y', strtotime($expire_date));
    $this_year = date('Y');
    $calc = abs($this_year - $year_expire);

    if ($calc >= 3) {
        $expire_date = 'Lifetime';
    }

    $subject = 'Your Purchase of Codarab Payment Smart Solutions - ' . $name;

    $html_template = "
    <p>Dear Customer,</p>

    <p>Thank you for purchasing the Codarab Payment Solution Plugin for WooCommerce! We're thrilled to have you on board and are excited for you to experience the benefits it brings to your online store.</p>

    <p>To install and activate the plugin, please follow these simple steps:</p>
    <p><strong>Before installing the Codarab Payment Plugin, please make sure you have removed all related PayPal plugins from your website. Otherwise, it can cause bugs on your site.</strong></p>
    <h3>Installation:</h3>
    <p>Please follow the instructions on the PDF link below</p>
   
    <p><a href='https://drive.google.com/file/d/1DrCKb-r4Yef_SbvrP4GczhX35TvImFTZ/preview' target='_blank'>PDF Installation Guide</a></p>

    <p>Activation code: {$activation_code}</p>
    <p>Your activation code for the plugin is valid until {$expire_date}. Please make sure to activate it before this date to ensure uninterrupted service.</p>

    <p>If you encounter any issues during installation or activation, please don't hesitate to contact our support team at <a href='https://codarab.com' target='_blank'>codarab.com</a>. We're here to assist you every step of the way.</p>

    <p>Once again, thank you for choosing Codarab Payment Smart Solutions. We hope it enhances your online store's functionality and provides a seamless payment experience for your customers.</p>

    <p>Best regards,</p>
    <p>CODARAB PAYMENT SMART SOLUTIONS</p>
    ";

    $headers = array('Content-Type: text/html; charset=UTF-8');
    wp_mail($customer_email, $subject, $html_template, $headers);
}
